##############################################################################
##
## Load-Zen-Datatools.ps1 [install|remove}
##
## by Actian Corporation (http://www.actian.com/zen)
##
##############################################################################

    <#
    .NOTES
        Copyright (c) Actian Corporation.  All rights reserved.

        Use of this script is subject to the terms of the Actian Zen license
        agreement under which you licensed the Zen product.  If you did not
        accept the terms of the license agreement, you are not authorized to
        use this script. For the terms of the license, please see the license
        agreement between you and Actian Corporation.
        THE SCRIPT IS PROVIDED "AS IS", WITH NO WARRANTIES.

    .SYNOPSIS
        Integrates the Data Tools for the PSQL ADO.NET Provider v4.4 into
        Visual Studio 2017 or later. Also removes the integration from Visual
        Studio 2017 or later.

    .DESCRIPTION
        Locates the latest installed version of Visual Studio (after VS 2017)
        using the vswhere.exe utility then either installs or removes the
        appropriate Visual Studio Extension package for the Data Tools for PSQL
        ADO.NET Provider v4.4. If Visual Studio is configured for Entity Framework
        development, also copies or deletes the Entity Framework Model First
        templates for the PSQL ADO.NET Provider v4.4 to/from the appropriate
        Visual Studio extensions folder. 

    .PARAMETER install
        Uses the VSIXInstaller.exe utility to install the appropriate PSQL ADO.NET
        Provider Data Tools VSIX package. Also copies the Entity Framework Model
        First templates to the appropriate Visual Studio 2017 or later installation
        location if Visual Studio is configured for Entity Framework development.
        A log file for the VSIXInstaller utility output is created in the %TEMP%
        folder with the name "zen.datatools.vsix.install.log".

    .PARAMETER remove
        Uses the VSIXInstaller.exe utility to uninstall the PSQL ADO.NET Provider
        Data Tools VSIX package. Also deletes the Entity Framework Model First
        templates from the Visual Studio 2017 or later installation location if
        Visual Studio is configured for Entity Framework development. A log file
        for the VSIXInstaller utility output is created in the %TEMP% folder with
        the name "zen.datatools.vsix.uninstall.log".

    .EXAMPLE
        PS C:\> .Load-Zen-Datatools.ps1 install
            Installs the VSIX package and copies the Entity Framework Model First
            templates to Visual Studio 2017 or later.

    .EXAMPLE
        PS C:\> .\Load-Zen-Datatools.ps1 remove
            Removes the installed VSIX package and deletes the Entity Framework Model
            First templates from Visual Studio 2017 or later.

    .INPUTS
        None. You cannot pipe objects to this cmdlet.

    .OUTPUTS
        0 : The script completed with no errors (may contain warnings).
        -1: The script returned one or more errors.

    #>

#Requires -version 3.0

param (
    [Parameter(
        Mandatory=$true,
        Position = 0)]
    [ValidateSet('install','remove')]
    [String]
    $InstallMode
)

Set-StrictMode -Version Latest

Import-LocalizedData msg -FileName "Load-Zen-DataTools.psd1" -ErrorAction Ignore

## Check for administrator rights without using the -RunAsAdministrator parameter for the #Requires
## directive (which requires PowerShell 4 or later).
$CurrentUser = New-Object -TypeName Security.Principal.WindowsPrincipal -ArgumentList $([Security.Principal.WindowsIdentity]::GetCurrent())
if ( ! $CurrentUser.IsInRole([Security.Principal.WindowsBuiltinRole]::Administrator)) {Throw $msg.errMustBeAdmin}

## Global variable - exists as long as PoSh session is running.
$global:ZenInfo=""

## Script-level variables definitions.
$scriptName = & {$myInvocation.ScriptName}
$scriptPath = & {Split-Path $myInvocation.ScriptName}
$registryPath = "HKLM:\SOFTWARE\Actian\Zen"
$supportedZenMajorVersionLevel = "14"
$vsixPkgName = Join-Path $scriptPath 'Zen.VisualStudio.DataTools_2017_2019.vsix'
$vsixId='PSQLDataTools.d115023a-8951-4aa5-ae36-94d7b1e6f6ad'
$modelFirstIncNames='GeneratePSQL.Utility*.ttinclude' #ModelFirst Include Template Names
$modelFirstT4Names='SSDLtoPSQL*.tt' # ModelFirst T4 Template Names
$modelFirstIncludes = join-path $scriptPath "Templates\ModelFirstTemplates\$modelFirstIncNames"
$modelFirstT4 = join-path $scriptPath "Templates\ModelFirstTemplates\$modelFirstT4Names"
$vswhere=join-path $scriptPath 'util\vswhere.exe'
$relPathVSIXInstaller='Common7\IDE\VSIXInstaller.exe'
$relPathEntityFrameworkTools='Common7\IDE\Extensions\Microsoft\Entity Framework Tools'
$relPathEntityFrameworkIncludes=join-path $relPathEntityFrameworkTools 'Templates\Includes'
$relPathEntityFrameworkT4=join-path $relPathEntityFrameworkTools 'DBGen'

function Main {
    ##################################################################################################
    ## Invoked as entry-point function at end of script and defines the order the other defined
    ## functions are called.
    ##################################################################################################
    if ($InstallMode -eq "install") {
        FindProductType
        CheckZenVersion
    }
    GetVSInfo
    RunVsixInstaller
    CopyOrRemoveTemplates
}

function FindProductType {
    # NOTE: This function is an exact duplicate of the same function in the winuwp Zen Edit removal
    # script 'Uninstall-Zen.ps1' (starting below this line).
    ##########################
    # Sets the installed product type based on one the 'HKLM:\SOFTWARE\Actian\Zen\InstallInfo'
    # sub-keys for Client, Server, Workgroup.
    #
    # Variables set:
    #   [string] $script:InstalledProductType = client|server|reporting
    #   [object] $global:ZenInfo = object methods match InstallInfo registry values.
    #                   $ZenInfo.InstallData
    #                   $ZenInfo.InstallDir64
    #                   $ZenInfo.InstallDir
    #                   $ZenInfo.IsBtrieve
    #                   $ZenInfo.IsReport
    #                   $ZenInfo.IsVx
    #                   $ZenInfo.ProductVersion
    #                   $ZenInfo.VersionLevel
    #                   $ZenInfo.InstallData
    #
    # Note: Must be called before logging is started.
    ##########################
    $script:InstalledProductType=""
    if (test-path $registryPath"\InstallInfo\") {
        Get-ChildItem $registryPath"\InstallInfo\" | ForEach-Object -Process {
        Write-Host "Name: "  $_
            switch ($_.PSChildName) {
                'Client' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$_"
                    $script:InstalledProductType="client"
                    break}
                'Server' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$_"
                    if ($ZenInfo.IsVx -eq 0) {
                        $script:InstalledProductType="server"
                    } else {
                        $script:InstalledProductType="vx"
                    }
                    break}
                'Workgroup' {
                    $global:ZenInfo=Get-ItemProperty $registryPath"\InstallInfo\$_"
                    if ($ZenInfo.IsReport -eq 0) {
                        $script:InstalledProductType="workgroup"
                    } else {
                        $script:InstalledProductType="reporting"
                    }
                    break}
            }
        }
    }
}

function CheckZenVersion {
    ##################################################################################################
    ## Verifies the major version specified in $ZenInfo.ProductVersion (global object set by previous
    ## call to FindProductType) starts with the major version number defined for by script variable
    ## $supportedZenMajorVersionLevel. The script is aborted if the detected major version is not
    ## the same as the defined major version.
    ##################################################################################################
    if ($ZenInfo -ne "") {
        $msg.infoZenInfo -f $ZenInfo.ProductVersion| Write-Output
        $currentVersion=$ZenInfo.ProductVersion[0]+$ZenInfo.ProductVersion[1]
        if ("$supportedZenMajorVersionLevel" -gt $currentVersion ) {
           $msg.errWrongZenVer -f $supportedZenMajorVersionLevel | Write-Error;exit -1
        }
    } else {
        $msg.errNoZenFound | Write-Error;exit -1
    }
}

function GetVSInfo {
    ##################################################################################################
    ## Creates the script level PoSh object "vsInfo" with the member properties specified below. If
    ## Visual Studio 2017 or later is not detected the script displays an error message and aborts.
    ## If the retrieved path to the VSIXInstaller utility is invalid the script also displays an error
    ## message and aborts.
    ##
    ## object member properties:
    ##    vsInfo.installDir    : value of property 'installationPath' returned by vswhere utility.
    ##    vsInfo.model1_inc    : path to Entity Framework Tools\Templates\Includes folder if it exists,
    ##                           empty string if it does not exist.
    ##    vsInfo.model1_t4     : path to Entity Framework Tools\DBGen folder if it exists, empty string
    ##                           if it does not exist.
    ##    vsInfo.version       : value of property 'installationVersion' returned by vswhere utility.
    ##    vsInfo.vsixInstaller : path to the VSIXInstaller utility (must exist).
    ##################################################################################################
    $vsInstallDirFound=$false
     
    ## Get installed version and location of VS. Vswhere requires the '-legacy' option for VS
    ## versions prior to VS 2017 (v15.0.#####.##).
    $vsVersion=& $vswhere -nologo -latest -products * -property installationVersion
    if ($vsVersion) {
        ## VS 2017 or later.
        $vsInstallDir=& $vswhere -nologo -latest -products * -property installationPath
        if ($vsInstallDir -and (Test-Path -Path $vsInstallDir)) {$vsInstallDirFound=$True}
    }

    ## Abort script if $vsVersion is not "15.0" or later (Visual Studio 2017 or later) or if the 
    ## installation path does not exist.
    if ( (!$vsVersion) -or ($vsVersion -lt "15.0") -or (!$vsInstallDirFound) ) {$msg.errNoVSFound -f "2017"| Write-Error;exit -1}

    ## Verify the location of the VSIXInstaller utility required to install/remove the VSIX package.
    $vsixInstaller=Join-Path $vsInstallDir $relPathVSIXInstaller
    if (! (test-path $vsixInstaller)) {$msg.errNoVSIXInstaller | Write-Error;exit -1}
     
    ## Verify the location of the Entity Framework Tools "Templates\Includes" and "DBGen" folders
    ## (sets corresponding vsInfo properties to empty string if the folder doesn't exist).
    $entityFrameworkIncludesPath=Join-Path $vsInstallDir $relPathEntityFrameworkIncludes
    if (!(test-path $entityFrameworkIncludesPath)) {$entityFrameworkIncludesPath=""}
    $entityFrameworkDBGenPath=Join-Path $vsInstallDir $relPathEntityFrameworkT4
    if (!(test-path $entityFrameworkDBGenPath)) {$entityFrameworkDBGenPath=""}

    ## Create PoSh object vsInfo.
    $script:vsInfo = @{
        'installDir'=$vsInstallDir;
        'model1_inc'=$entityFrameworkIncludesPath;
        'model1_t4'=$entityFrameworkDBGenPath;
        'version'=$vsVersion;
        'vsixInstaller'=$vsixInstaller;
    }
     
    ## Output VS version and location info.
    $msg.infoVisualStudioInfo -f $vsInfo.version,$vsInfo.installDir | Write-Output
}

function RunVsixInstaller {
    ##################################################################################################
    ## Runs VSIXInstaller utility to install or remove the appropriate VSIX package.  If the
    ## VSIXInstaller.exe process cannot be started an error message is displayed and the script is
    ## aborted, otherwise an informational message instructing the user to check the VSIXInstaller log
    ## file is displayed.
    ##################################################################################################

    ## Set the VSIX package name and install/uninstall options based on Visual Studio version.
    $vsixInstallArgs="/quiet","/logFile:Zen.DataTools.VSIX.install.log","$vsixPkgName"
    $vsixInstallMsg=$msg.infoInstalledVsixPkgLatestWithLog -f [string]"Zen.DataTools.VSIX.install.log"
    $vsixInstallErr=$msg.errVsixInstallWithLog -f [string]" (Zen.DataTools.VSIX.install.log)"
    $vsixRemovalArgs="/quiet","/logFile:Zen.DataTools.VSIX.uninstall.log","/uninstall:$vsixId"
    $vsixRemovalMsg=$msg.infoRemovedVsixPkgLatestWithLog -f [string]"Zen.DataTools.VSIX.uninstall.log"
    $vsixRemovalErr=$msg.errVsixUninstallWithLog -f [string]" (Zen.DataTools.VSIX.uninstall.log)"


    ## Install or remove the VSIX package based on $InstallMode.
    if ($InstallMode -eq "install") {
        Try {
           $msg.infoInstallingVsixPkg | Write-Output
           Start-Process -FilePath $vsInfo.vsixInstaller -ArgumentList $vsixInstallArgs -NoNewWindow -Wait
           $vsixInstallMsg | Write-Output
        }
        Catch {$vsixInstallErr | Write-Error;exit -1}
    } else {
        Try {
            $msg.infoRemovingVsixPkg | Write-Output
            Start-Process -FilePath $vsInfo.vsixInstaller -ArgumentList $vsixRemovalArgs -NoNewWindow -Wait
            $vsixRemovalMsg | Write-Output
        }
        Catch {$vsixRemovalErr | Write-Error;exit -1}
    } 
}

function CopyOrRemoveTemplates {
    ##################################################################################################
    ## Copies or removes the Entity Framework Model First templates to/from the appropriate Visual
    ## Studio folder.  If the path to the Entity Framework Tools folder cannot be located this step
    ## is skipped.
    ##################################################################################################

    if ($InstallMode -eq "install") {  ## Copy templates.
        Try {
            ## Verify Entity Framework Tools folder and copy templates.
            $entityFrameworkTools=join-path $vsInfo.installDir $relPathEntityFrameworkTools
            if (Test-Path $entityFrameworkTools) {

                ## Model First include templates.
                Copy-Item -Path $modelFirstIncludes -Destination $vsInfo.model1_inc -Force
                if ($?) {$msg.infoAddedModelFirstIncludeTemplates | Write-Output}

                ## Model First T4 templates.
                Copy-Item -Path $modelFirstT4 -Destination $vsInfo.model1_t4 -Force
                if ($?) {$msg.infoAddedModelFirstT4Templates | Write-Output}

            }
        }
        Catch {$msg.warnNoTemplates -f [string]"$relPathEntityFrameworkTools" | Write-Warning}
    } else {  ## Remove templates.
        Try {
            ## Verify Model First include templates exist and remove them.
            $removeIncludes=join-path $vsInfo.model1_inc $modelFirstIncNames
            if (Test-Path $removeIncludes) {
                remove-item -Path $removeIncludes -Force
                if ($?) {$msg.infoRemovedModelFirstIncludeTemplates | Write-Output}
            }

            ## Verify Model First T4 templates exist and remove them.
            $removeT4=join-path $vsInfo.model1_t4 $modelFirstT4Names
            if (Test-Path $removeT4) {
                remove-item -Path $removeT4 -Force
                if ($?) {$msg.infoRemovedModelFirstT4Templates | Write-Output}
            }
        }
        Catch {$msg.warnNoTemplates -f [string]"$relPathEntityFrameworkTools" | Write-Warning}
    }
}

## Invoke Main function.
"" | Write-Output
$msg.infoRunning -f $MyInvocation.InvocationName, $InstallMode | Write-Output
. Main
$msg.infoEnd -f $MyInvocation.InvocationName, $InstallMode | Write-Output
"" | Write-Output
exit 0
# SIG # Begin signature block
# MIIcMwYJKoZIhvcNAQcCoIIcJDCCHCACAQExCzAJBgUrDgMCGgUAMGkGCisGAQQB
# gjcCAQSgWzBZMDQGCisGAQQBgjcCAR4wJgIDAQAABBAfzDtgWUsITrck0sYpfvNR
# AgEAAgEAAgEAAgEAAgEAMCEwCQYFKw4DAhoFAAQU41kLcrGZ8stg+FQ4dW/3p5P+
# fZ+gghcYMIID7jCCA1egAwIBAgIQfpPr+3zGTlnqS5p31Ab8OzANBgkqhkiG9w0B
# AQUFADCBizELMAkGA1UEBhMCWkExFTATBgNVBAgTDFdlc3Rlcm4gQ2FwZTEUMBIG
# A1UEBxMLRHVyYmFudmlsbGUxDzANBgNVBAoTBlRoYXd0ZTEdMBsGA1UECxMUVGhh
# d3RlIENlcnRpZmljYXRpb24xHzAdBgNVBAMTFlRoYXd0ZSBUaW1lc3RhbXBpbmcg
# Q0EwHhcNMTIxMjIxMDAwMDAwWhcNMjAxMjMwMjM1OTU5WjBeMQswCQYDVQQGEwJV
# UzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xMDAuBgNVBAMTJ1N5bWFu
# dGVjIFRpbWUgU3RhbXBpbmcgU2VydmljZXMgQ0EgLSBHMjCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBALGss0lUS5ccEgrYJXmRIlcqb9y4JsRDc2vCvy5Q
# WvsUwnaOQwElQ7Sh4kX06Ld7w3TMIte0lAAC903tv7S3RCRrzV9FO9FEzkMScxeC
# i2m0K8uZHqxyGyZNcR+xMd37UWECU6aq9UksBXhFpS+JzueZ5/6M4lc/PcaS3Er4
# ezPkeQr78HWIQZz/xQNRmarXbJ+TaYdlKYOFwmAUxMjJOxTawIHwHw103pIiq8r3
# +3R8J+b3Sht/p8OeLa6K6qbmqicWfWH3mHERvOJQoUvlXfrlDqcsn6plINPYlujI
# fKVOSET/GeJEB5IL12iEgF1qeGRFzWBGflTBE3zFefHJwXECAwEAAaOB+jCB9zAd
# BgNVHQ4EFgQUX5r1blzMzHSa1N197z/b7EyALt0wMgYIKwYBBQUHAQEEJjAkMCIG
# CCsGAQUFBzABhhZodHRwOi8vb2NzcC50aGF3dGUuY29tMBIGA1UdEwEB/wQIMAYB
# Af8CAQAwPwYDVR0fBDgwNjA0oDKgMIYuaHR0cDovL2NybC50aGF3dGUuY29tL1Ro
# YXd0ZVRpbWVzdGFtcGluZ0NBLmNybDATBgNVHSUEDDAKBggrBgEFBQcDCDAOBgNV
# HQ8BAf8EBAMCAQYwKAYDVR0RBCEwH6QdMBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0y
# MDQ4LTEwDQYJKoZIhvcNAQEFBQADgYEAAwmbj3nvf1kwqu9otfrjCR27T4IGXTdf
# plKfFo3qHJIJRG71betYfDDo+WmNI3MLEm9Hqa45EfgqsZuwGsOO61mWAK3ODE2y
# 0DGmCFwqevzieh1XTKhlGOl5QGIllm7HxzdqgyEIjkHq3dlXPx13SYcqFgZepjhq
# IhKjURmDfrYwggR9MIIDZaADAgECAgMb5xUwDQYJKoZIhvcNAQELBQAwYzELMAkG
# A1UEBhMCVVMxITAfBgNVBAoTGFRoZSBHbyBEYWRkeSBHcm91cCwgSW5jLjExMC8G
# A1UECxMoR28gRGFkZHkgQ2xhc3MgMiBDZXJ0aWZpY2F0aW9uIEF1dGhvcml0eTAe
# Fw0xNDAxMDEwNzAwMDBaFw0zMTA1MzAwNzAwMDBaMIGDMQswCQYDVQQGEwJVUzEQ
# MA4GA1UECBMHQXJpem9uYTETMBEGA1UEBxMKU2NvdHRzZGFsZTEaMBgGA1UEChMR
# R29EYWRkeS5jb20sIEluYy4xMTAvBgNVBAMTKEdvIERhZGR5IFJvb3QgQ2VydGlm
# aWNhdGUgQXV0aG9yaXR5IC0gRzIwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEK
# AoIBAQC/cWII8fpZNPcbyRij94BJWOkigxOmxSBDATuE8eaFSZ8n6vaEG06gtNtw
# mMcyAbEFPgdO7vT6Ty9ZMCLnqxlWa+KAB/zzFnWAOVF75fk1tnROqY2CE+S2P6kD
# g/qivooVan/eC8O2GRQFyurDqASUO0Z8Mg3zAGYiyI1pbTaMERi307IcYLQ4+gKM
# ztPdRgfeCj7rXXzIfPuwK1OkkmJpUSUFYRpEgYwsqUOWI9+sOoGaDinFHKnpXR62
# np4wCjnO8YiA+0tdzDLshWJDJTQCVicBkbQ7cCo/brHonIgBfZ/U+dtTbWCdvyzn
# WKu4X0b8zsQbAzwJ60kxXGlGs+BHAgMBAAGjggEXMIIBEzAPBgNVHRMBAf8EBTAD
# AQH/MA4GA1UdDwEB/wQEAwIBBjAdBgNVHQ4EFgQUOpqFBxBnKLbv9r0FQW4gwZTa
# D94wHwYDVR0jBBgwFoAU0sSw0pHUTBFxs2HLPaH+3ahq1OMwNAYIKwYBBQUHAQEE
# KDAmMCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5nb2RhZGR5LmNvbS8wMgYDVR0f
# BCswKTAnoCWgI4YhaHR0cDovL2NybC5nb2RhZGR5LmNvbS9nZHJvb3QuY3JsMEYG
# A1UdIAQ/MD0wOwYEVR0gADAzMDEGCCsGAQUFBwIBFiVodHRwczovL2NlcnRzLmdv
# ZGFkZHkuY29tL3JlcG9zaXRvcnkvMA0GCSqGSIb3DQEBCwUAA4IBAQBZC1O9koYR
# pyR77Vsxzx0fbHDFuG6+Trv2vpdQ4TB/uihcYpTC434z9/tCdoXblRyMIlh1CQyI
# ZWc5ChYJxaA4l6TFI5M/tBimAQZEkeOnaSe0WiV/Orcyzd2E/yo4KTOk3Weyhf6h
# iCAcUInI3Cr2QgM3TOaI39WvJPKxw9/MtezgmV63SVQgPJQYDMccUhhJpG3hs1gL
# ydjs2a4cMo4ocA3i/qYXnoQPvVdws1rpH6CGU7vvfP9pC+BIw7eTC8gKVMSsXRRn
# N2zKpS8xCDeqbm+MvJviV10kga+Xl5yErWysN0xm82GRESDkvjCfeqQpCbDhNF9k
# dxhAUd+MMKavMIIEozCCA4ugAwIBAgIQDs/0OMj+vzVuBNhqmBsaUDANBgkqhkiG
# 9w0BAQUFADBeMQswCQYDVQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9y
# YXRpb24xMDAuBgNVBAMTJ1N5bWFudGVjIFRpbWUgU3RhbXBpbmcgU2VydmljZXMg
# Q0EgLSBHMjAeFw0xMjEwMTgwMDAwMDBaFw0yMDEyMjkyMzU5NTlaMGIxCzAJBgNV
# BAYTAlVTMR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3JhdGlvbjE0MDIGA1UEAxMr
# U3ltYW50ZWMgVGltZSBTdGFtcGluZyBTZXJ2aWNlcyBTaWduZXIgLSBHNDCCASIw
# DQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBAKJjCzlEuLsjp0RJuw7/ofBhClOT
# sJjbrSwPSsVu/4Y8U1UPFc4EPyv9qZaW2b5heQtbyUyGduXgQ0sile7CK0PBn9ho
# tI5AT+6FOLkRxSPyZFjwFTJvTlehroikAtcqHs1L4d1j1ReJMluwXplaqJ0oUA4X
# 7pbbYTtFUR3PElYLkkf8q672Zj1HrHBy55LnX80QucSDZJQZvSWA4ejSIqXQugJ6
# oXeTW2XD7hd0vEGGKtwITIySjJEtnndEH2jWqHR32w5bMotWizO92WPISZ06xcXq
# MwvS8aMb9Iu+2bNXizveBKd6IrIkri7HcMW+ToMmCPsLvalPmQjhEChyqs0CAwEA
# AaOCAVcwggFTMAwGA1UdEwEB/wQCMAAwFgYDVR0lAQH/BAwwCgYIKwYBBQUHAwgw
# DgYDVR0PAQH/BAQDAgeAMHMGCCsGAQUFBwEBBGcwZTAqBggrBgEFBQcwAYYeaHR0
# cDovL3RzLW9jc3Aud3Muc3ltYW50ZWMuY29tMDcGCCsGAQUFBzAChitodHRwOi8v
# dHMtYWlhLndzLnN5bWFudGVjLmNvbS90c3MtY2EtZzIuY2VyMDwGA1UdHwQ1MDMw
# MaAvoC2GK2h0dHA6Ly90cy1jcmwud3Muc3ltYW50ZWMuY29tL3Rzcy1jYS1nMi5j
# cmwwKAYDVR0RBCEwH6QdMBsxGTAXBgNVBAMTEFRpbWVTdGFtcC0yMDQ4LTIwHQYD
# VR0OBBYEFEbGaaMOShQe1UzaUmMXP142vA3mMB8GA1UdIwQYMBaAFF+a9W5czMx0
# mtTdfe8/2+xMgC7dMA0GCSqGSIb3DQEBBQUAA4IBAQB4O7SRKgBM8I9iMDd4o4Qn
# B28Yst4l3KDUlAOqhk4ln5pAAxzdzuN5yyFoBtq2MrRtv/QsJmMz5ElkbQ3mw2cO
# 9wWkNWx8iRbG6bLfsundIMZxD82VdNy2XN69Nx9DeOZ4tc0oBCCjqvFLxIgpkQ6A
# 0RH83Vx2bk9eDkVGQW4NsOo4mrE62glxEPwcebSAe6xp9P2ctgwWK/F/Wwk9m1vi
# FsoTgW0ALjgNqCmPLOGy9FqpAa8VnCwvSRvbIrvD/niUUcOGsYKIXfA9tFGheTMr
# Lnu53CAJE3Hrahlbz+ilMFcsiUk/uc9/yb8+ImhjU5q9aXSsxR08f5Lgw7wc2AR1
# MIIE0DCCA7igAwIBAgIBBzANBgkqhkiG9w0BAQsFADCBgzELMAkGA1UEBhMCVVMx
# EDAOBgNVBAgTB0FyaXpvbmExEzARBgNVBAcTClNjb3R0c2RhbGUxGjAYBgNVBAoT
# EUdvRGFkZHkuY29tLCBJbmMuMTEwLwYDVQQDEyhHbyBEYWRkeSBSb290IENlcnRp
# ZmljYXRlIEF1dGhvcml0eSAtIEcyMB4XDTExMDUwMzA3MDAwMFoXDTMxMDUwMzA3
# MDAwMFowgbQxCzAJBgNVBAYTAlVTMRAwDgYDVQQIEwdBcml6b25hMRMwEQYDVQQH
# EwpTY290dHNkYWxlMRowGAYDVQQKExFHb0RhZGR5LmNvbSwgSW5jLjEtMCsGA1UE
# CxMkaHR0cDovL2NlcnRzLmdvZGFkZHkuY29tL3JlcG9zaXRvcnkvMTMwMQYDVQQD
# EypHbyBEYWRkeSBTZWN1cmUgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IC0gRzIwggEi
# MA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQC54MsQ1K92vdSTYuswZLiBCGzD
# BNliF44v/z5lz4/OYuY8UhzaFkVLVat4a2ODYpDOD2lsmcgaFItMzEUz6ojcnqOv
# K/6AYZ15V8TPLvQ/MDxdR/yaFrzDN5ZBUY4RS1T4KL7QjL7wMDge87Am+GZHY23e
# cSZHjzhHU9FGHbTj3ADqRay9vHHZqm8A29vNMDp5T19MR/gd71vCxJ1gO7GyQ5HY
# pDNO6rPWJ0+tJYqlxvTV0KaudAVkV4i1RFXULSo6Pvi4vekyCgKUZMQWOlDxSq7n
# eTOvDCAHf+jfBDnCaQJsY1L6d8EbyHSHyLmTGFBUNUtpTrw700kuH9zB0lL7AgMB
# AAGjggEaMIIBFjAPBgNVHRMBAf8EBTADAQH/MA4GA1UdDwEB/wQEAwIBBjAdBgNV
# HQ4EFgQUQMK9J47MNIMwojPX+2yz8LQsgM4wHwYDVR0jBBgwFoAUOpqFBxBnKLbv
# 9r0FQW4gwZTaD94wNAYIKwYBBQUHAQEEKDAmMCQGCCsGAQUFBzABhhhodHRwOi8v
# b2NzcC5nb2RhZGR5LmNvbS8wNQYDVR0fBC4wLDAqoCigJoYkaHR0cDovL2NybC5n
# b2RhZGR5LmNvbS9nZHJvb3QtZzIuY3JsMEYGA1UdIAQ/MD0wOwYEVR0gADAzMDEG
# CCsGAQUFBwIBFiVodHRwczovL2NlcnRzLmdvZGFkZHkuY29tL3JlcG9zaXRvcnkv
# MA0GCSqGSIb3DQEBCwUAA4IBAQAIfmyTEMg4uJapkEv/oV9PBO9sPpyIBslQj6Zz
# 91cxG7685C/b+LrTW+C05+Z5Yg4MotdqY3MxtfWoSKQ7CC2iXZDXtHwlTxFWMMS2
# RJ17LJ3lXubvDGGqv+QqG+6EnriDfcFDzkSnE3ANkR/0yBOtg2DZ2HKocyQetawi
# DsoXiWJYRBuriSUBAA/NxBti21G00w9RKpv0vHP8ds42pM3Z2Czqrpv1KrKQ0U11
# GIo/ikGQI31bS/6kA1ibRrLDYGCD+H1QQc7CoZDDu+8CL9IVVO5EFdkKrqeKM+2x
# LXY2JtwE65/3YR8V3Idv7kaWKK2hJn0KCacuBKONvPi8BDABMIIFJjCCBA6gAwIB
# AgIICsxCB6AJxjswDQYJKoZIhvcNAQELBQAwgbQxCzAJBgNVBAYTAlVTMRAwDgYD
# VQQIEwdBcml6b25hMRMwEQYDVQQHEwpTY290dHNkYWxlMRowGAYDVQQKExFHb0Rh
# ZGR5LmNvbSwgSW5jLjEtMCsGA1UECxMkaHR0cDovL2NlcnRzLmdvZGFkZHkuY29t
# L3JlcG9zaXRvcnkvMTMwMQYDVQQDEypHbyBEYWRkeSBTZWN1cmUgQ2VydGlmaWNh
# dGUgQXV0aG9yaXR5IC0gRzIwHhcNMTcwNDA1MjAwMjAwWhcNMjAwNDEwMTk1MTM5
# WjBoMQswCQYDVQQGEwJVUzEOMAwGA1UECBMFVGV4YXMxDzANBgNVBAcTBkF1c3Rp
# bjEbMBkGA1UEChMSQWN0aWFuIENvcnBvcmF0aW9uMRswGQYDVQQDExJBY3RpYW4g
# Q29ycG9yYXRpb24wggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQDDl5bZ
# gsKcnWzrY8rYrsSmI4l9tC5Ipk1Dp/LHYZQ/1uwgzbiUQavCOKhJ6q+XMGHIOF6P
# N/PkfIfrf1XygR3bJy+f0u4HICioNg0GmfBgEUHVpZTkeRxCT+Dd1FeDDljp9J41
# AVkXh8SnzKvSGKjaUbp3DkH2Y5P24pHR9zhpx2hnqSF0PHOT0fGFcF4SAeYBTIT0
# nSmVPxjs2+NV1M81xtCLLLCF9ceV784DyY7MhoHI9Q7fHpxUJZi0nuHRzyUurofC
# QmJhOLd5vm7re/WUAHfhG3P7dIkm1mI9SHAWid2A/4vGWHb/bR5mwJTDxvPmgfzz
# EVmR8PHPNTsRXZYpAgMBAAGjggGFMIIBgTAMBgNVHRMBAf8EAjAAMBMGA1UdJQQM
# MAoGCCsGAQUFBwMDMA4GA1UdDwEB/wQEAwIHgDA1BgNVHR8ELjAsMCqgKKAmhiRo
# dHRwOi8vY3JsLmdvZGFkZHkuY29tL2dkaWcyczUtMi5jcmwwXQYDVR0gBFYwVDBI
# BgtghkgBhv1tAQcXAjA5MDcGCCsGAQUFBwIBFitodHRwOi8vY2VydGlmaWNhdGVz
# LmdvZGFkZHkuY29tL3JlcG9zaXRvcnkvMAgGBmeBDAEEATB2BggrBgEFBQcBAQRq
# MGgwJAYIKwYBBQUHMAGGGGh0dHA6Ly9vY3NwLmdvZGFkZHkuY29tLzBABggrBgEF
# BQcwAoY0aHR0cDovL2NlcnRpZmljYXRlcy5nb2RhZGR5LmNvbS9yZXBvc2l0b3J5
# L2dkaWcyLmNydDAfBgNVHSMEGDAWgBRAwr0njsw0gzCiM9f7bLPwtCyAzjAdBgNV
# HQ4EFgQUd28uU+LsLPArCXvwS49GX5IaKcUwDQYJKoZIhvcNAQELBQADggEBAKMz
# CNStFW8r/jaaSDXwbWnpmRjeGx3FZbmwH+r2gv2UQ5Se0ckdRm86CfF8AjGnFd/d
# pOSsLWUSWx5+gdqorB9+xY6KrB0br08CfRYvvw+ZYiRrnvLEJyD3hc13tHYpmDOS
# lamg3seFzk/rKe5g/ZXwDQXOiAMEkH5xBHS3BBILQarbiJ14CLM9eLDSeN8L/Gdy
# SvMX4nWy9UusiXP3ZXvuVPUlAOBvDUZaM1BPir0ZWY/m2FOW8bHxXzTG8VtdgY0y
# JIkaii7O3A1kkB9ofB47LQN/CuwcOOKQx0hNL3Nlq7zsFizWk6MO3ZmPNjjimkI6
# mT0NuoY6yd9VefcO95MxggSFMIIEgQIBATCBwTCBtDELMAkGA1UEBhMCVVMxEDAO
# BgNVBAgTB0FyaXpvbmExEzARBgNVBAcTClNjb3R0c2RhbGUxGjAYBgNVBAoTEUdv
# RGFkZHkuY29tLCBJbmMuMS0wKwYDVQQLEyRodHRwOi8vY2VydHMuZ29kYWRkeS5j
# b20vcmVwb3NpdG9yeS8xMzAxBgNVBAMTKkdvIERhZGR5IFNlY3VyZSBDZXJ0aWZp
# Y2F0ZSBBdXRob3JpdHkgLSBHMgIICsxCB6AJxjswCQYFKw4DAhoFAKCBijAZBgkq
# hkiG9w0BCQMxDAYKKwYBBAGCNwIBBDAcBgorBgEEAYI3AgELMQ4wDAYKKwYBBAGC
# NwIBFTAjBgkqhkiG9w0BCQQxFgQULk6hIFC09HVIdHe3n9Idowy+e2QwKgYKKwYB
# BAGCNwIBDDEcMBqhGIAWaHR0cDpcXHd3dy5hY3RpYW4uY29tIDANBgkqhkiG9w0B
# AQEFAASCAQAZCN2q/mZ2JszbHdMUffSzq+eu0apqPP0rlM9hKNdn+h1bpbdgZE6C
# ACPkgYMglWH3rSwWenMaVurJViXNz6COcYP/X6T233dQaZ7sNjnHW+GJoqpJkNYF
# 3IT+Gi7J9Jg3DAfYC00NsdHd6i3aVH5dPjikwOW9wEx2OA76f37btI4iVyx+x6Vt
# l0xNz9b80uqwDg0R7/Qg7+/URPQUB8oCBEQQEeVr5WOOB8ra0LwgQHYi+z8AOxGP
# Hdwex0igaVW04rTRQpHEsp6PzoGkeXA3JFin0PXhTnl86rTEjNJUEwdH9G84qBie
# 0Ek5e3SL3xfnmAs5vqwToDYzNkYKOYjzoYICCzCCAgcGCSqGSIb3DQEJBjGCAfgw
# ggH0AgEBMHIwXjELMAkGA1UEBhMCVVMxHTAbBgNVBAoTFFN5bWFudGVjIENvcnBv
# cmF0aW9uMTAwLgYDVQQDEydTeW1hbnRlYyBUaW1lIFN0YW1waW5nIFNlcnZpY2Vz
# IENBIC0gRzICEA7P9DjI/r81bgTYapgbGlAwCQYFKw4DAhoFAKBdMBgGCSqGSIb3
# DQEJAzELBgkqhkiG9w0BBwEwHAYJKoZIhvcNAQkFMQ8XDTIwMDQwMTE2MjUyN1ow
# IwYJKoZIhvcNAQkEMRYEFKnXT+fhB0EDpxZd1BvlvBwSwGZUMA0GCSqGSIb3DQEB
# AQUABIIBAAMmRY+Tnk5M73W0WAkXmB/nHS7jD6wN7hOs+6Ls1JsXBtKpcwCAfxkn
# /LInldIrIFr4ANIkZh58x0IqZTT8OSuHkkyaTkvH7tsf5snn2CeChN/aLlOMES8I
# l2IyVZ6RVrNNC8Rx7qJMls8lq8xi7Sb1kFxnBdE8NfSDqr7i7gcgpa/347JQWD1X
# YDp+WZLYuLwCjEvIryRaD6O6ko3mfzFEMYPlcenNRXKQvSCShgh4Mrtc9qr0YlXc
# JZ1RKVa5FaytLFKq/s+EGcSuhqkV+ziM/pkRRKKy5IKKXi/HkfiVT/RtdzG4HzXC
# m83rgfeQpxtUlUFgD0g0GO1Z/K7emwA=
# SIG # End signature block
